// -------------------------------------------------------------------------------------------------------------------------
// Module Name: LPS Ethernet Socket Communication Device Control Linux Library
// NB 09/05/2025    Initial Version
// NB 09/29/2025    Added enhanced status reports for new devices
// -------------------------------------------------------------------------------------------------------------------------
// This library uses .05 degree units for attenuation values, so 10 degrees is represented by 200. (multiply by 20 to convert degrees to api units)
/// ---------- Include headers ----------------
#include <stdbool.h>
#include <stdint.h>         
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <time.h>
#include <errno.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <arpa/inet.h> 
#include <netdb.h>
#include <poll.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <sys/timerfd.h>
#include <sys/wait.h> 
#include "lpssocket.h"

/// ---------- Macros ----------------
#define _CRT_SECURE_CPP_OVERLOAD_SECURE_NAMES 	1
#define _CRT_SECURE_NO_WARNINGS					1

#define LPS_SOCKET_PORT  "40001"
#define FALSE 0
#define TRUE 1          
#define PACKET_CTRL_LEN 8
#define PACKET_INT_LEN  8

#define LIBVER "1.1"
#define LPS_LIBVERSION 0x101        // we return an integer representation of the version with one byte each for major and minor version

// IP Address Validator
#define DELIM "."

// Socket connetion timeout
#define CONNECTION_TIMEOUT_SEC 5

// Command Send Throttling Macros
#define TCP_SEND_INTERVAL 200       // This is the default delayed ACK time
#define MAX_COMMANDS_PER_WINDOW 48

// Thread States Macros
#define THREAD_IDLE 0
#define THREAD_START 1
#define THREAD_EXIT 3
#define THREAD_DEAD 4
#define THREAD_ERROR -1


/* -----------------------------------------------------------------
Global Data to maintain the Devices data & states for each process
-------------------------------------------------------------------*/
bool TestMode = FALSE;          // if TestMode is true we fake it -- no HW access
int Trace_Out = 0;          // variable to control how much tracing to do
int IO_Trace = 0;           // used for tracing control during device init and I/O operations
bool bVerbose = FALSE; 		// True to generate debug oriented printf output during device enumeration
bool status_empty = FALSE; // True if the status report from the device was empty

// to force LPS library debugging messages choose the following definitions
#define DEBUG_OUT Trace_Out     /* set this to 1 in order to see debugging output, 2 for a ton of output, or 3 for many tons */

int  lpsdevcount = 0;
time_t starttime;
LPSPARAMS lps[MAXDEVICES];      // an array of structures each of which holds the info for a given device

/* stuff for throttling */
pthread_mutex_t throttle_locks[MAXDEVICES] = { [0 ... MAXDEVICES-1] = PTHREAD_MUTEX_INITIALIZER };
int win_command_counts[MAXDEVICES] = {0};

static struct timespec prev_cmd_times[MAXDEVICES] = {0};

/* stuff for the threads */
pthread_t threads[MAXDEVICES];


/// ---------- Const Defintions----------
const char* sVNX_devicenames[] = { "LPS-802-8", "LPS-402-8", "LPS-202-8" };

char STATUS_QUERY_ID[]= {VNX_GETSERNUM, VNX_MAXCHANNEL, VNX_SWVERSION, VNX_MODELNAME, VNX_MINFREQUENCY, VNX_MAXFREQUENCY,
                         VNX_MINPHASE, VNX_MAXPHASE, VNX_CHANNEL, VNX_FREQUENCY, VNX_PHASEANGLE, VNX_HR8STATUS, VNX_ASTART, VNX_ASTOP,
                         VNX_ASTEP, VNX_ASTEP2, VNX_ADWELL, VNX_ADWELL2, VNX_AIDLE, VNX_AHOLD,
                         VNX_PROFILECOUNT, VNX_PROFILEDWELL, VNX_PROFILEIDLE, VNX_SWEEP, VNX_IPMODE, VNX_IPADDR, VNX_IPMASK, VNX_IPGW};


char LibVersion[] = LIBVER;

/* globals we'll be using at runtime */
char errmsg[32]; 			// For the status->string converter

// Forward declarations
static void *rxdatahandler_callback(void *threadID);
bool SendReport(DEVID deviceID, unsigned char command, unsigned char *pBuffer, int cbBuffer);
bool RawSend(DEVID deviceID, unsigned char command, unsigned char *pBuffer, int cbBuffer);
bool GetParameter(DEVID deviceID, int GetParam);
bool GetDeviceStatus(DEVID deviceID);
int SetChannel(DEVID deviceID, int channel);
bool CheckV2Features(DEVID deviceID);



/// ----------  Static Fucntion Calls ----------------
//***************************************************************************
//
// Function call for returning the DLL Library Version
//
//*****************************************************************************
int fnLPS_GetLibVersion(void){
    return LPS_LIBVERSION;
}

//***************************************************************************
//
// Function call for Settig into test mode
//
//*****************************************************************************
void fnLPS_SetTestMode(bool testmode) {
    TestMode = testmode;
}

//***************************************************************************
//
// A function to display the status as string
//
//*****************************************************************************
char* fnLPS_perror(LVSTATUS status) {
  strcpy(errmsg, "STATUS_OK");
  if (BAD_PARAMETER == status) strcpy(errmsg, "BAD_PARAMETER");
  if (BAD_HID_IO == status) strcpy(errmsg, "BAD_HID_IO");
  if (DEVICE_NOT_READY == status) strcpy(errmsg, "DEVICE_NOT_READY");
  if (FEATURE_NOT_SUPPORTED == status) strcpy(errmsg, "FEATURE_NOT_SUPPORTED");
  if (DATA_UNAVAILABLE == status) strcpy(errmsg, "DATA_UNAVAILABLE");
  if (DEV_TABLE_FULL == status) strcpy(errmsg, "DEV_TABLE_FULL");
  if (BAD_IP_ADDRESS == status) strcpy(errmsg, "BAD_IP_ADDRESS");

  // Status returns for DevStatus
  if (INVALID_DEVID == status) strcpy(errmsg, "INVALID_DEVID");
  if (DEV_CONNECTED == status) strcpy(errmsg, "DEV_CONNECTED");
  if (DEV_OPENED == status) strcpy(errmsg, "DEV_OPENED");
  if (SWP_ACTIVE == status) strcpy(errmsg,  "SWP_ACTIVE");
  if (SWP_UP == status) strcpy(errmsg, "SWP_UP");
  if (SWP_REPEAT == status) strcpy(errmsg, "SWP_REPEAT");
  if (SWP_BIDIRECTIONAL == status) strcpy(errmsg, "SWP_BIDIRECTIONAL");
  if (PROFILE_ACTIVE == status) strcpy(errmsg, "PROFILE_ACTIVE");

  return errmsg;

}

bool CheckV2Features(DEVID deviceID) {
  // Even for multichannel devices, we can use channel 0 to know if the device is V2
  if (lps[deviceID].DevStatus[0] & DEV_V2FEATURES)
    {
      return TRUE;
    }
  else return FALSE;
}

bool CheckHiRes(DEVID deviceID) {
  // Even for multichannel devices, we can use channel 0 to know if the device is HiRes
  if (lps[deviceID].DevStatus[0] & DEV_HIRES)
    {
      return TRUE;
    }
  else return FALSE;
}

//***************************************************************************
//
// Function call for delay loop
//
//*****************************************************************************
/* usleep is deprecated, so this is a function using nanosleep() */
void sleep_ms(long naptime) {
  // naptime comes in as the number of ms we want. 20 = 20ms
  struct timespec timeval;
   timeval.tv_sec = 0;
   timeval.tv_nsec = naptime * 1000000L;

   nanosleep(&timeval , NULL);
}

//***************************************************************************
//
// Wait for the device's buffer to be empty
//
//*****************************************************************************
bool wait_for_empty_buffer(int devid) {
    bool result;
    const int MAX_RETRIES = 10;
    int retries = 0;
    BYTE VNX_param[6] = {0, 0, 0, 0, 0, 0};

    result = GetDeviceStatus(devid);
    while (((lps[devid].DevStatus[0] & DEV_BUFEMPTY) == 0 || !result || status_empty) && retries++ < MAX_RETRIES) {
        sleep_ms(TCP_SEND_INTERVAL);
        result = GetDeviceStatus(devid);
    }

    if (retries >= MAX_RETRIES) {
        if (DEBUG_OUT > 0) {
            printf("Warning: Timeout waiting for buffers to empty on device %d\n", devid);
            return false;
        }
    }

    return true;
}

//***************************************************************************
//
// Throttle the number of commands sent to the device if we are sending too many
//
//****************************************************************************
bool maybe_throttle(int devid) {
    int cur_ms;
    int prev_ms;
    struct timespec ts;

    // Get the current time
    clock_gettime(CLOCK_MONOTONIC, &ts);
    // Calculate the time in ms of the start of the window versus current time in ms
    prev_ms = (prev_cmd_times[devid].tv_sec * 1000) + (prev_cmd_times[devid].tv_nsec / 1000000);
    cur_ms = (ts.tv_sec * 1000) + (ts.tv_nsec / 1000000);

    if (cur_ms - prev_ms > TCP_SEND_INTERVAL) {
        // Reset the window
        win_command_counts[devid] = 0;
    } else if (win_command_counts[devid] > MAX_COMMANDS_PER_WINDOW && prev_ms != 0) {
        // The maximum command count has been hit for this window
        sleep_ms(TCP_SEND_INTERVAL);
        // Check if the buffer is empty
        // We can only do this on V2 devices where we have status updates
        if (CheckV2Features(devid)) {
            if (!wait_for_empty_buffer(devid)) return false;
        }

        win_command_counts[devid] = 0;
    }

    // Keep track of the time for this command send
    clock_gettime(CLOCK_MONOTONIC, &ts);
    prev_cmd_times[devid] = ts;
    
    // Increment the command counter
    win_command_counts[devid]++;
    return true;
}

//***************************************************************************
//
// Validate netconfig digits and return 1 if string contain only digits, else return 0
//
//*****************************************************************************
int valid_digit(char *ip_str){
    while (*ip_str) {
        if (*ip_str >= '0' && *ip_str <= '9')
            ++ip_str;
        else
            return 0;
    }
    return 1;
}

//***************************************************************************
//
// Validate IP Configuration -return 1 if IP string is valid, else return 0
//
//*****************************************************************************
int is_valid_ip(char *ip_str){
    int num, dots = 0;
    char *ptr;
    char lstr[16];
    char* next_token;
    strcpy(lstr, ip_str);

    if (lstr == NULL)
        return 0;

    ptr = strtok(lstr, DELIM);

    if (ptr == NULL)
        return 0;

    while (ptr) {

        /* after parsing string, it must contain only digits */
        if (!valid_digit(ptr))
            return 0;

        num = atoi(ptr);

        /* check for valid IP */
        if (num >= 0 && num <= 255) {
            /* parse remaining string */
            ptr = strtok(NULL, DELIM);
            if (ptr != NULL)
                ++dots;
        } else
            return 0;
    }

    /* valid IP string must contain 3 dots */
    if (dots != 3)
        return 0;
    return 1;
}

//***************************************************************************
//
// Function call to Get DeviceID based on the deviceip from the device entry list
//
//*****************************************************************************
int GetDeviceID(char* deviceip){
    int index=-1;

    // Loop through to get the device index
    for(index=0; index < MAXDEVICES; index++)
    {
        if(strcmp(lps[index].deviceip, deviceip)==0)
            return index;
    }

    return index;
}

//***************************************************************************
//
// Function call to check whether LPS device socket open or not
//
//*****************************************************************************
bool isDeviceSockOpen(DEVID deviceID) {
    if (TestMode) return TRUE;// in test mode all devices are always available

    //  printf("devid:%d\n",devid);
    if(deviceID >=0)
    {
        //      printf("devicesockfd:%d\n",lps[devid].devicesockfd);
        if (lps[deviceID].devicesockfd != 0)
            return TRUE;
        else
            return FALSE;
    }
    else
        return FALSE;
}

//***************************************************************************
//
// Function check if the device is locked or not
//
//*****************************************************************************
bool DevNotLocked(DEVID deviceID)
{
    if (TestMode) return true;        // this shouldn't happen, but just in case...

    if (!(lps[deviceID].DevStatus[0] & DEV_LOCKED))
    {
        return true;                // we return true if the device is not locked!
    }
    else return false;
}

//***************************************************************************
//
// Function call to lock the device
//
//*****************************************************************************
void LockDev(DEVID deviceID, bool lock)
{
    if (TestMode) return;        // this shouldn't happen, but just in case...

    if (lock)
    {
        lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] | DEV_LOCKED;
        return;
    }
    else
    {
        lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] & ~DEV_LOCKED;
        return;
    }
}

// -------------- Get Routines to read device settings --------------------
//
// Note: for these functions deviceID is not checked for validity
//		 since it was already checked in the calling program.

bool GetDeviceStatus(DEVID deviceID) {
  // Raw send because we don't want to throttle
  BYTE VNX_param[6] = { 0, 0, 0, 0, 0, 0 };
  if (!RawSend(deviceID, VNX_HR8STATUS, VNX_param, 0)) {
      return FALSE;
  }
  return TRUE;
}

bool GetPhaseAngle(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PHASEANGLE))
    return FALSE;

  return TRUE;
}

bool GetFrequency(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_FREQUENCY))
    return FALSE;

  return TRUE;
}

// -------------------------------

bool GetIdleTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_AIDLE))
    return FALSE;

  return TRUE;
}

// -------------------------------

bool GetHoldTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_AHOLD))
    return FALSE;

  return TRUE;
}
// -------------------------------

bool GetRampStart(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTART))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetRampStep(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetRampStep2(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP2))
    return FALSE;

  return TRUE;
}
// -------------------------------

bool GetRampEnd(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTOP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetDwellTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ADWELL))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetDwellTime2(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ADWELL2))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetWorkingFrequency(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_FREQUENCY))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetRampStop(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTOP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetPhaseAngleStep(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetPhaseAngleStep2(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP2))
    return FALSE;

  return TRUE;
}

// ---- profile related Get functions ------
bool GetProfileCount(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILECOUNT))
    return FALSE;

  return TRUE;
}

bool GetProfileDwellTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILEDWELL))
    return FALSE;

  return TRUE;
}

bool GetProfileIdleTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILEIDLE))
    return FALSE;

  return TRUE;
}

// ---- profile related Get functions ------
bool GetProfileIndex(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILEINDEX))
      return FALSE;
  return TRUE;
}

bool GetChannel(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_CHANNEL))
    return FALSE;

  return TRUE;
}


// RD -- Revised to use a single element cache, see the response parser for details
bool GetProfileElement(DEVID deviceID, int index)
{
  int max_index;
	unsigned int bank;
	unsigned int ch;
	int timedout;

	max_index = lps[deviceID].ProfileMaxLength - 1;

	if (max_index < 0 || max_index > PROFILE_MAX_RAM - 1) lps[deviceID].ProfileMaxLength = 50;	// this should never happen, if it does, pick the smallest profile size for safety

    if (index < 0 || index > max_index) return FALSE;    // bail out on a bad index...

	ch = lps[deviceID].Channel;
	if (ch >= CHANNEL_MAX) ch = 0;				// just defending our arrays, should never happen

	bank = (unsigned int)(ch / 8) + 1;		// the logical bank is incremented before it is transmitted
											// this lets us determine if we have a logical bank in the command packet


  BYTE VNX_command[] = {0, 0, 0, 0, 0, 0};

  VNX_command[2] = (BYTE)(index & 0x00FF);				// Index low byte
  VNX_command[3] = (BYTE)((index >> 8) & 0x03);			// Index high bits (only bits 0–1 used)
  VNX_command[4] = (BYTE)bank;							// Logical bank number
  VNX_command[5] = (BYTE)lps[deviceID].ChMask;			// Logical channel mask

  // --- now we send the command out to the device --
  if (!SendReport(deviceID, VNX_SETPROFILE | VNX_GET, VNX_command, 6))
      return FALSE;

	return TRUE;
}



bool GetSerNum(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_GETSERNUM))
    return FALSE;

  return TRUE;
}

//***************************************************************************
//
// Function call to check whether LPS device socket open or not
//
//*****************************************************************************
static int vnxOpenDevice(char* deviceip) {
    int sockfd = -1;
    struct addrinfo hints;
    struct addrinfo *result, *rp;
    int flags;
    int iResult;

    memset(&hints, 0, sizeof(struct addrinfo));
    hints.ai_canonname = NULL;
    hints.ai_addr = NULL;
    hints.ai_next = NULL;
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = AI_NUMERICSERV;

    if (DEBUG_OUT > 1) printf("IP:%s\n",deviceip);
 
    // Convert IPv4 and IPv6 addresses from text to binary form 
    if(inet_pton(AF_INET, deviceip, &hints.ai_addr)<=0)  
    { 
        if (DEBUG_OUT > 0) printf("\nInvalid address/ Address not supported \n"); 
        return DEVICE_NOT_READY; 
    } 

    // Check if Device is already open or not
    if (getaddrinfo(deviceip, LPS_SOCKET_PORT, &hints, &result) != 0)
        return DEVICE_NOT_READY;

    for (rp = result; rp != NULL; rp = rp->ai_next)
    {
        //  Open the Socket Connection 
        sockfd = socket(rp->ai_family, rp->ai_socktype, rp->ai_protocol);
        if (sockfd == -1)
            continue;

        // Set socket to non-blocking
        flags = fcntl(sockfd, F_GETFL, 0);
        if (flags == -1) {
            close(sockfd);
            continue;
        }
        if (fcntl(sockfd, F_SETFL, flags | O_NONBLOCK) == -1) {
            close(sockfd);
            continue;
        }

        iResult = connect(sockfd, rp->ai_addr, rp->ai_addrlen);
        if (iResult < 0) {
            if (errno != EINPROGRESS) {
                close(sockfd);
                sockfd = -1;
                continue;
            }
        }

        if (iResult < 0) {
            fd_set writefds;
            FD_ZERO(&writefds);
            FD_SET(sockfd, &writefds);

            struct timeval tv;
            tv.tv_sec = CONNECTION_TIMEOUT_SEC;
            tv.tv_usec = 0;

            iResult = select(sockfd + 1, NULL, &writefds, NULL, &tv);
            if (iResult <= 0) {
                close(sockfd);
                sockfd = -1;
                continue;
            }

            // Check if the connection actually succeeded
            int optVal;
            socklen_t optLen = sizeof(optVal);
            if (getsockopt(sockfd, SOL_SOCKET, SO_ERROR, &optVal, &optLen) < 0 || optVal != 0) {
                close(sockfd);
                sockfd = -1;
                continue;
            }
        }

        // Restore blocking mode
        if (fcntl(sockfd, F_SETFL, flags & ~O_NONBLOCK) == -1) {
            close(sockfd);
            sockfd = -1;
            continue;
        }

        //Successfully connected
        break;
    }

    if (rp == NULL)
    {
        if (sockfd != -1) close(sockfd);
        return DEVICE_NOT_READY;
    }

    freeaddrinfo(result);

    return sockfd;
}

//***************************************************************************
//
// -- our main function to find LPS attached devices --
//
//***************************************************************************
int FindVNXDevices(void)
{
    int i;
    int j;
    int NumDevices = 0;             // we will total up how many LPS devices we find
    int sockfd;
    int err;
    int validModel = 0;

    for (i = 1; i < MAXDEVICES; i++)
    {
        if (is_valid_ip(lps[i].deviceip)) {
            // Try to open a socket to the device
            sockfd = vnxOpenDevice(lps[i].deviceip);
            if (sockfd > 0) {
                lps[i].devicesockfd = sockfd;

                // Create a device threads for reading the data from the device using sockets
                err = pthread_create(&threads[i], NULL, rxdatahandler_callback, (void*)(uintptr_t)i);

                // Query some basic parameters
                if (GetParameter(i, VNX_GETSERNUM) && GetParameter(i, VNX_MODELNAME) &&
                    GetParameter(i, VNX_MAXFREQUENCY) && GetParameter(i, VNX_MINFREQUENCY) &&
                    GetParameter(i, VNX_IPADDR) && GetParameter(i, VNX_IPMODE) && 
                    GetParameter(i, VNX_IPGW) && GetParameter(i, VNX_IPMASK)) {
                    // Loop through each model name to check if this device is one we want
                    for (int j = 0; j < (sizeof(sVNX_devicenames) / sizeof(sVNX_devicenames[0])); j++) {
                        if (strcmp(lps[i].ModelName, sVNX_devicenames[j]) == 0) {
                            // We found one of our devices
                            validModel = 1;
                            break;
                        }
                    }
                    if (validModel) {
                        lps[i].DevStatus[0] |= DEV_CONNECTED;
                        lps[i].DevStatus[0] |= DEV_OPENED;
                        NumDevices++;
                    }
                } else {
                    lps[i].DevStatus[0] &= ~DEV_CONNECTED;
                }
            } else {
                lps[i].DevStatus[0] &= ~DEV_CONNECTED;
            }
            
            // Close down the socket connection
            close(sockfd);
            lps[i].devicesockfd = 0;
            lps[i].DevStatus[0] &= ~DEV_OPENED;    // Remove the dev opened flag
        } else {
            lps[i].DevStatus[0] &= ~DEV_CONNECTED;
        }
    }

    return NumDevices;
}

//***************************************************************************
//
// Function call to Send the VNX commands through the Socket to the given ip
//  device
//*****************************************************************************
bool SendReport(DEVID deviceID, unsigned char command, unsigned char *pBuffer, int cbBuffer)
{
    int index;
    int timedout;

    // Check if we are sending too many commands to the device
    // This will wait a little bit if so
    pthread_mutex_lock(&throttle_locks[deviceID]);
    if (!maybe_throttle(deviceID)) {
        // Something went wrong with throttling, don't send!
        pthread_mutex_unlock(&throttle_locks[deviceID]);
        return FALSE;
    }
    pthread_mutex_unlock(&throttle_locks[deviceID]);

    // Make sure the buffer that is being passed to us fits
    if (cbBuffer > HR_BLOCKSIZE) {
        // Report too big, don't send!
        return FALSE;
    }

    // lets make sure our command doesn't have any junk bits in it
    for (index=0; index<HID_REPORT_LENGTH; index++)
        lps[deviceID].sndbuff[index] = 0;

    if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%d\r\n", (uint8_t)command, cbBuffer);
    lps[deviceID].sndbuff[0] = command;       // command to device
    lps[deviceID].sndbuff[1] = cbBuffer;
    for (index=0; index<cbBuffer; index++)
        lps[deviceID].sndbuff[index+2] = pBuffer[index];

    if (DEBUG_OUT > 1) {
    printf("SR to device %d: ", deviceID);
    for (index=0; index<8; index++) {
        printf("%02x ", (uint8_t)lps[deviceID].sndbuff[index]);
    }
    printf("\r\n");
    }

    lps[deviceID].responseready = 0; // Clear Device Response before command send

    // Call Socket Function to send packet data to the device
    if (write(lps[deviceID].devicesockfd, lps[deviceID].sndbuff, PACKET_CTRL_LEN) < 0 )
        if (DEBUG_OUT > 0) printf("sending Message Error!");

    //Check for Get /Set Message type and expect response if Get message only
    if((command &  VNX_SET) == 0)
    {
        // --- then we wait for the read thread's parser to signal that it got the response
        starttime = time(NULL);
        timedout = 0;
        
        // wait until the value is decoded or 1 seconds have gone by
        // RD 4-2019 modified to yield the thread during the wait
        while ((!lps[deviceID].responseready) && (0 == timedout)) {
        sleep_ms(10);   // yield for 10 ms, speed matters here
        if ((time(NULL)-starttime) > 1) {
            if (DEBUG_OUT > 0) printf("Timeout waiting for response to %x command", command);
            timedout = 1;
        }
        }

        return (0 == timedout) ? TRUE : FALSE;
    }
    else
        return TRUE;
}

//***************************************************************************
//
// Minimal, fast send with no throttling
//
//*****************************************************************************
bool RawSend(DEVID deviceID, unsigned char command, unsigned char *pBuffer, int cbBuffer)
{
    int index;
    int timedout;

    // Make sure the buffer that is being passed to us fits
    if (cbBuffer > HR_BLOCKSIZE) {
        // Report too big, don't send!
        return FALSE;
    }

    // lets make sure our command doesn't have any junk bits in it
    for (index=0; index<HID_REPORT_LENGTH; index++)
        lps[deviceID].sndbuff[index] = 0;

    if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%d\r\n", (uint8_t)command, cbBuffer);
    lps[deviceID].sndbuff[0] = command;       // command to device
    lps[deviceID].sndbuff[1] = cbBuffer;
    for (index=0; index<cbBuffer; index++)
        lps[deviceID].sndbuff[index+2] = pBuffer[index];

    if (DEBUG_OUT > 1) {
    printf("SR to device %d: ", deviceID);
    for (index=0; index<8; index++) {
        printf("%02x ", (uint8_t)lps[deviceID].sndbuff[index]);
    }
    printf("\r\n");
    }

    lps[deviceID].responseready = 0; // Clear Device Response before command send

    // Call Socket Function to send packet data to the device
    if (write(lps[deviceID].devicesockfd, lps[deviceID].sndbuff, PACKET_CTRL_LEN) < 0 )
        if (DEBUG_OUT > 0) printf("sending Message Error!");

    //Check for Get /Set Message type and expect response if Get message only
    if((command &  VNX_SET) == 0)
    {
        // --- then we wait for the read thread's parser to signal that it got the response
        starttime = time(NULL);
        timedout = 0;
        
        // wait until the value is decoded or 1 seconds have gone by
        // RD 4-2019 modified to yield the thread during the wait
        while ((!lps[deviceID].responseready) && (0 == timedout)) {
        sleep_ms(10);   // yield for 10 ms, speed matters here
        if ((time(NULL)-starttime) > 1) {
            if (DEBUG_OUT > 0) printf("Timeout waiting for response to %x command", command);
            timedout = 1;
        }
        }

        return (0 == timedout) ? TRUE : FALSE;
    }
    else
        return TRUE;
}

//***************************************************************************
//
// Function call to clear Cache for the device data structures with default data
//  
//*****************************************************************************
void ClearDevCache(DEVID deviceID, int num_channels) {
	int ch;
	int i;

	if (num_channels > CHANNEL_MAX) num_channels = CHANNEL_MAX;	// protect our arrays

	for (ch = 0; ch < num_channels; ch++) {
		lps[deviceID].PhaseAngle[ch] = -1;			// we mark the phase angle as unknown to start
		lps[deviceID].WorkingFrequency[ch] = -1;	// and everybody else...
		lps[deviceID].RampStart[ch] = -1;
		lps[deviceID].RampStop[ch] = -1;
		lps[deviceID].PhaseStep[ch] = -1;
		lps[deviceID].PhaseStep2[ch] = -1;
		lps[deviceID].DwellTime[ch] = -1;
		lps[deviceID].DwellTime2[ch] = -1;
		lps[deviceID].IdleTime[ch] = -1;
		lps[deviceID].HoldTime[ch] = -1;
		lps[deviceID].Modebits[ch] = 0;				// Modebits aren't directly readable, but clearing them here for convenience
		lps[deviceID].ProfileIndex[ch] = -1;
		lps[deviceID].ProfileDwellTime[ch] = -1;
		lps[deviceID].ProfileIdleTime[ch] = -1;
		lps[deviceID].ProfileCount[ch] = -1;
		lps[deviceID].CachedProfileValue[ch] = -1;	// The int holds the index and the value, we init the whole thing to 0xFFFFFFFF to indicate it is empty.
	}
}

void decode_status(unsigned char swpdata, int deviceID, int ch) {

	lps[deviceID].DevStatus[ch] &= ~(SWP_ACTIVE | SWP_REPEAT | SWP_BIDIRECTIONAL | PROFILE_ACTIVE);	// preset all the zero bit cases

	// are we ramping?
	if (swpdata & (SWP_ONCE | SWP_CONTINUOUS))
		lps[deviceID].DevStatus[ch] = lps[deviceID].DevStatus[ch] | SWP_ACTIVE;

     // -- fill in the SWP_UP status bit
	if (swpdata & (SWP_DIRECTION))  // are we ramping downwards?
		lps[deviceID].DevStatus[ch] = lps[deviceID].DevStatus[ch] & ~SWP_UP;
	else
		lps[deviceID].DevStatus[ch] = lps[deviceID].DevStatus[ch] | SWP_UP;

	// -- fill in the continuous sweep bit
	if (swpdata & (SWP_CONTINUOUS))  // are we in continuous ramp mode?
		lps[deviceID].DevStatus[ch] = lps[deviceID].DevStatus[ch] | SWP_REPEAT;

	// -- fill in the bidirectional ramp bit
	if (swpdata & (SWP_BIDIR))  // are we in bi-directional ramp mode?
		lps[deviceID].DevStatus[ch] = lps[deviceID].DevStatus[ch] | SWP_BIDIRECTIONAL;

	// -- fill in the profile active bit
	if (swpdata & (STATUS_PROFILE_ACTIVE))  // is a phase profile playing?
		lps[deviceID].DevStatus[ch] = lps[deviceID].DevStatus[ch] | PROFILE_ACTIVE;

}


/*******************************************************************************
 * \brief    Parse Data Frame received from the socket
 *
 ******************************************************************************/
int parseDataFrame(char* msg, int count, int sockfd, int devid)
{
    int i;
    int chnl;
    RESPONSE_DATA_T dataresp;
    unsigned short dataval_16;
	unsigned short prfindex;
    unsigned int dataval_32;
    unsigned char dataval_8;
	char *dataptr;
	char namebuffer[7];

#if 0
    for(i=0; i < count; i++)
    {
        if (DEBUG_OUT > 1) printf("%02X",(unsigned char)msg[i]);
    }

    if (DEBUG_OUT > 1) printf("Parse Data Frame:(Sockfd-%d, Devid-%d, Count-%d)\n",sockfd,devid,count);
#endif

    dataresp.command = msg[0];	            // Command ID
    dataresp.length = msg[1];	              // Data Length
    dataptr = &msg[2];			                // Data buffer Start
    dataval_16 = *(unsigned short*)&msg[2];	// Data of 16 bits
    dataval_32 = *(unsigned int*)&msg[2];	  // Data of 32 bits
    dataval_8 =  *(unsigned char*)&msg[2];	// Data of 8bits 

    // The reports don't carry a channel address so we use the global channel setting
    if ((lps[devid].HaveNumChannels) || (lps[devid].NumChannels != 0))
    {	
      if(lps[devid].Channel < lps[devid].NumChannels)
          chnl = lps[devid].Channel;
      else
        chnl = lps[devid].NumChannels - 1;
    }
    else
    {
      chnl = 0;	// we don't know how many channels the device has yet
    }

    switch(dataresp.command)
    {
        case VNX_HR8STATUS:
            if (DEBUG_OUT > 2) printf("VNX_HR8STATUS: decoding\r\n");

            if (DevNotLocked(devid)) {
                // Only V2 devices (firmware version >= 1.1.00) have enhanced status reports
                int major = 0, minor = 0, patch = 0;
                if (sscanf(lps[devid].Swversion, "%d.%d.%d", &major, &minor, &patch) == 3) {
                    if ((major > 0 && minor >= 1) || major >= 2) {
                        lps[devid].DevStatus[0] |= DEV_V2FEATURES;      // Just in case this wasnt set earlier

                        dataresp.length = msg[1] & 0x3F;    // Use only the lower 6 bits, the upper two contain buffer status info
                        if (dataresp.length != 0) {    // Check to make sure that we don't have an error status report
                            status_empty = FALSE;   // This is a valid status report

                            // Check if buffers are empty (msg[1] is the length + status byte, we need to check the two highest bits)
                            if ((msg[1] & 0xC0) == 0) {
                                lps[devid].DevStatus[0] = lps[devid].DevStatus[0] | DEV_BUFEMPTY;
                                if (DEBUG_OUT > 2) printf("Device %d: Buffers are empty\n", devid);
                            }
                            else {
                                lps[devid].DevStatus[0] = lps[devid].DevStatus[0] & ~DEV_BUFEMPTY;
                                if (DEBUG_OUT > 2) printf("Device %d: Buffers are NOT empty\n", devid);
                            }

                            // defend against a bad channel number
                            if (chnl >= CHANNEL_MAX) chnl = CHANNEL_MAX - 1;

                            lps[devid].PhaseAngle[chnl] = (int)msg[5] + 256 * (int)msg[6];	// update the HiRes phase angle level (in .05 degree units already)
                            
                            if (DEBUG_OUT > 2) printf("  VNX_HR8STATUS reports Phase Angle=%d\r\n", lps[devid].PhaseAngle[chnl]);
                            
                            // -- extract and save the status bits
                            decode_status(msg[7], devid, chnl);
                            
                            if (DEBUG_OUT > 2) printf("  VNX_HR8STATUS sez Channel %d Status=%02x\r\n", chnl + 1, lps[devid].DevStatus[chnl]);
                        
                        }
                        else
                        {
                            status_empty = TRUE;    // This is not a valid status report
                            if (DEBUG_OUT > 2) printf("VNX_HR8STATUS Error status report of length 0\r\n");
                        }
                    }
                }
            }
            break;

        case VNX_PHASEANGLE:
            if (DEBUG_OUT > 0) printf(" Phase Angle = %d\n", dataval_16);
            if (DevNotLocked(devid)) {
                lps[devid].PhaseAngle[chnl] = dataval_16 * lps[devid].UnitScale;
            }
            break;

        case VNX_FREQUENCY:
            if (DEBUG_OUT > 0) printf(" Working Frequency = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].WorkingFrequency[chnl] = dataval_32;
            }
            break;

        case VNX_SWEEP:
            if (DEBUG_OUT > 0) {
                if (dataval_16)
                    printf("%s\n", "Sweep ON");
                else
                    printf("%s\n", "Sweep OFF");
            }
            if (DevNotLocked(devid)) {
                // merge in the ramp mode bits from the device
                lps[devid].Modebits[chnl] = lps[devid].Modebits[chnl] & (~0xff);
                lps[devid].Modebits[chnl] |= dataval_8;
            }
            break;

        case VNX_ASTART:
            if (DEBUG_OUT > 0) printf(" Ramp Start Phase Angle = %d\n", dataval_16 * lps[devid].UnitScale);
            if (DevNotLocked(devid)) {
            lps[devid].RampStart[chnl] = dataval_16 * lps[devid].UnitScale;
            }
            break;

        case VNX_ASTOP:
            if (DEBUG_OUT > 0) printf(" Ramp Stop Phase Angle = %d\n", dataval_16 * lps[devid].UnitScale);
            if (DevNotLocked(devid)) {
                lps[devid].RampStop[chnl] = dataval_16 * lps[devid].UnitScale;
            }
            break;

        case VNX_ASTEP:
            if (DEBUG_OUT > 0) printf(" Phase Step = %d\n", dataval_16 * lps[devid].UnitScale);
            if (DevNotLocked(devid)) {
                lps[devid].PhaseStep[chnl] = dataval_16 * lps[devid].UnitScale;
            }
            break;

        case VNX_ASTEP2:
            if (DEBUG_OUT > 0) printf(" Phase Step 2 = %d\n", dataval_16 * lps[devid].UnitScale);
            if (DevNotLocked(devid)) {
                lps[devid].PhaseStep2[chnl] = dataval_16 * lps[devid].UnitScale;
            }
            break;

        case VNX_ADWELL:
            if (DEBUG_OUT > 0) printf(" Dwell Time = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].DwellTime[chnl] = dataval_32;
            }
            break;

        case VNX_ADWELL2:
            if (DEBUG_OUT > 0) printf(" Dwell Time 2 = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].DwellTime2[chnl] = dataval_32;
            }
            break;

        case VNX_AIDLE:
            if (DEBUG_OUT > 0) printf(" Idle Time = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].IdleTime[chnl] = dataval_32;
            }
            break;

        case VNX_AHOLD:
            if (DEBUG_OUT > 0) printf(" Hold Time = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].HoldTime[chnl] = dataval_32;
            }
            break;

        case VNX_MINPHASE:
            if (DEBUG_OUT > 0) printf(" Min Phase = %d\n", dataval_16 * lps[devid].UnitScale);
            if (DevNotLocked(devid)) {
                lps[devid].MinPhase = dataval_16 * lps[devid].UnitScale;
            }
            break;

        case VNX_MAXPHASE:
            if (DEBUG_OUT > 0) printf(" Max Phase = %d\n", dataval_16 * lps[devid].UnitScale);
            if (DevNotLocked(devid)) {
                lps[devid].MaxPhase = dataval_16 * lps[devid].UnitScale;
            }
            break;

		  case VNX_MAXCHANNEL:
            if (DevNotLocked(devid)) {
                if (lps[devid].HaveNumChannels == 0)
                {
                    if (dataval_32 <= CHANNEL_MAX)
                    {
                        // we have a reasonable number of channels for this device
                        lps[devid].NumChannels = dataval_32;
                        lps[devid].HaveNumChannels = TRUE;
                    }
                }
                else
                {
                    if (dataval_32 != lps[devid].NumChannels)
                    {
                        if (DEBUG_OUT > 0) printf("Default MaxChannel = %d, new MaxChannel = %d\n", lps[devid].NumChannels, dataval_32);
                    }
                }
            }
			break;

        case VNX_GETSERNUM:
            if (DevNotLocked(devid)) {
                lps[devid].SerialNumber = dataval_32;
            }
            break;

			// The model name is used to identify the device type
			// The VNX_MODELNAME function returns the last 6 characters of the model name from
			// the device. We use that (or a part of it) to identify the device and set DevType
			// for non-expandable devices we set the maximum number of channels at this point
			//

        case VNX_MODELNAME:
          if (DEBUG_OUT > 1)
          {
            printf("In ModelName [packet]: ");
              for(i = 0; i < count; i++)
              {
                  printf("%02X ",(unsigned char)msg[i]);
              }
            printf("\n");
          }

            if (DevNotLocked(devid)) {
                // build a string from the 6 name bytes we received
                for(i = 0; i < 6; i++)
                {
                    namebuffer[i] = msg[2+i];
                }
                namebuffer[6] = 0;		

                if (strstr(namebuffer,"802-8"))
                {
                    strcpy (lps[devid].ModelName, "LPS-802-8");
                    lps[devid].DevType = LPS_802_8;
                    lps[devid].MaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
                    lps[devid].MinPhaseStep = 20;		// 1 degree steps
                    lps[devid].UnitScale = 1;
                    lps[devid].MinFrequency = 40000;	// 4000 MHz
                    lps[devid].MaxFrequency = 80000;	// 8000 MHz
                    lps[devid].Expandable = false;
                    lps[devid].NumChannels = 8;
                    lps[devid].BaseChannels = 8;
                    lps[devid].HaveNumChannels = TRUE;
                }
                else if (strstr(namebuffer,"402-8"))
                {
                    strcpy (lps[devid].ModelName, "LPS-402-8");
                    lps[devid].DevType = LPS_402_8;
                    lps[devid].MaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
                    lps[devid].MinPhaseStep = 20;		// 1 degree steps
                    lps[devid].UnitScale = 1;
                    lps[devid].MinFrequency = 20000;	// 2000 MHz
                    lps[devid].MaxFrequency = 40000;	// 4000 MHz
                    lps[devid].Expandable = false;
                    lps[devid].NumChannels = 8;
                    lps[devid].BaseChannels = 8;
                    lps[devid].HaveNumChannels = TRUE;
                }
                else if (strstr(namebuffer,"202-8"))
                {
                    strcpy (lps[devid].ModelName, "LPS-202-8");
                    lps[devid].DevType = LPS_202_8;
                    lps[devid].MaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
                    lps[devid].MinPhaseStep = 20;		// 1 degree steps
                    lps[devid].UnitScale = 1;
                    lps[devid].MinFrequency = 40000;	// 4000 MHz
                    lps[devid].MaxFrequency = 80000;	// 8000 MHz
                    lps[devid].Expandable = false;
                    lps[devid].NumChannels = 8;
                    lps[devid].BaseChannels = 8;
                    lps[devid].HaveNumChannels = TRUE;
                }
                else
                {
                    // RD!! - we should flag this error
                    strncpy (lps[devid].ModelName, namebuffer, 6);  // save the device's model name
                    lps[devid].MaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
                    lps[devid].MinPhaseStep = 20;		// 1 degree steps
                    lps[devid].UnitScale = 1;
                    lps[devid].MinFrequency = 40000;	// 4000 MHz
                    lps[devid].MaxFrequency = 80000;	// 8000 MHz
                    lps[devid].Expandable = false;
                    lps[devid].NumChannels = 8;
                    lps[devid].BaseChannels = 8;
                    lps[devid].HaveNumChannels = TRUE;
                }

                // set a few parameters for the device
                lps[devid].MinPhase = 0;
                lps[devid].ProfileMaxLength = 1000;
                lps[devid].ProfileMaxLength = PROFILE_MAX_RAM;
                lps[devid].ProfileMaxSaved = PROFILE_MAX_HR;
                lps[devid].DevStatus[0] |= DEV_HIRES;
            }

			if (DEBUG_OUT > 2) printf("Exiting ModelName...\n");
            break;

        case VNX_SWVERSION:
            if (DevNotLocked(devid)) {
                // build a string from the 6 bytes we received
                for(i = 0; i < 6; i++)
                {
                    lps[devid].Swversion[i] = msg[2+i];		// safe copy...
                }
                lps[devid].Swversion[6] = 0;				// the null terminator for the string	

                // Only V2 devices (firmware version >= 1.1.00) have enhanced status reports
                int major = 0, minor = 0, patch = 0;
                if (sscanf(lps[devid].Swversion, "%d.%d.%d", &major, &minor, &patch) == 3) {
                    if ((major > 0 && minor >= 1) || major >= 2) {
                        lps[devid].DevStatus[0] |= DEV_V2FEATURES;
                    }
                }
            }
            break;
            
        case VNX_IPMODE:
            if (DevNotLocked(devid)) {
                lps[devid].ipmode = (int)dataval_8;
            }
            break;

        case VNX_IPADDR:
            if (DevNotLocked(devid)) {
                sprintf(lps[devid].ipaddress, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                    (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            }
            break;

        case VNX_IPMASK:
            if (DevNotLocked(devid)) {
                sprintf(lps[devid].netmask, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                    (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            }
            break;

        case VNX_IPGW:
            if (DevNotLocked(devid)) {
                sprintf(lps[devid].gateway, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                    (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            }
            lps[devid].deviceready = 1; // Set Device Ready after init calls
            break;

        case VNX_MINFREQUENCY:
            if (DEBUG_OUT > 0) printf(" Min Working Frequency = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].MinFrequency = dataval_32;
            }
            break;

        case VNX_MAXFREQUENCY:
            if (DEBUG_OUT > 0) printf(" Max Working Frequency = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].MaxFrequency = dataval_32;
            }
            break;

        case VNX_CHANNEL:
            if (DevNotLocked(devid)) {
                if ((int)dataval_8 > 0) {
                    lps[devid].Channel = (int)dataval_8 - 1;	// The global channel number, 1 to NumChannels
                                                                // internally we use zero based channel numbers
                    if (lps[devid].Channel < 8) {
                        lps[devid].ChMask = 0x01 << lps[devid].Channel;
                    } else {
                        lps[devid].ChMask = 0x01 << (lps[devid].Channel % 8);
                    }
                    if ((lps[devid].Channel >= lps[devid].NumChannels) || (lps[devid].Channel < 0))
                    {
                        lps[devid].Channel = lps[devid].NumChannels - 1;	// clip to protect our arrays...
                    }
                } else {
                    lps[devid].Channel = 0;
                    lps[devid].ChMask = 0x01;
                }
            }

            if (DEBUG_OUT > 1) printf(" Channel %d of %d total channels\n", lps[devid].Channel, lps[devid].NumChannels);
            break;

        case VNX_PROFILECOUNT:
            if (DEBUG_OUT > 0) printf(" Profile Count = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].ProfileCount[chnl] = dataval_32;
            }
            break;

        case VNX_PROFILEDWELL:
            if (DEBUG_OUT > 0) printf(" Profile Dwell Time = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].ProfileDwellTime[chnl] = dataval_32;
            }
            break;

        case VNX_PROFILEIDLE:
            if (DEBUG_OUT > 0) printf(" Profile Idle Time = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].ProfileIdleTime[chnl] = dataval_32;
            }
            break;

        case VNX_SETPROFILE:
            if (DevNotLocked(devid)) {
                prfindex = *(unsigned short*)&msg[6];   // Data of 16 bits
                if (prfindex < PROFILE_MAX_RAM) 		// defend our array
                {
                    lps[devid].CachedProfileValue[chnl] = dataval_16 * lps[devid].UnitScale;
                }

                if (DEBUG_OUT > 2) printf(" profile element cache = %x\n", lps[devid].CachedProfileValue[chnl]);
                if (DEBUG_OUT > 1) printf(" profile element %d = %d\n", prfindex, lps[devid].CachedProfileValue[chnl] & 0xFFFF);
            }
            break;

        case VNX_PROFILEINDEX:
            if (DEBUG_OUT > 0) printf(" Profile Index = %d\n", dataval_32);
            if (DevNotLocked(devid)) {
                lps[devid].ProfileIndex[chnl] = dataval_32;
            }
            break;

        default:
            break;
    }

    lps[devid].responseready = 1;  // Device Response Received

    return STATUS_OK;
}

/*******************************************************************************
 * \brief    Reads Data from the socket
 ******************************************************************************/
int readdatafromSock(int sockfd, int devid)
{
    int count;
    char *buf_pr;
    char buf_st[512] = {0};
    int index;

    buf_pr = buf_st;
        
    if ((count = read(sockfd, buf_pr, 512)) >= 0) {

//        printf("READ FROM SOCKET(%d): Count-%d\n", sockfd, count);
        // Check Socket data response > 8 then split into 8 bytes chunks and parse the command data
        if(count > HID_REPORT_LENGTH)
        {
            for(index=0; index < count; index+=8)
            {
                if (DEBUG_OUT > 1) printf("Send Frame Data(%d,%d)\n",index, count);
                parseDataFrame(buf_pr+index, 8, sockfd, devid);
            }
        }
        else
            parseDataFrame(buf_pr, count, sockfd, devid);
    }
    return 0;
}

/*******************************************************************************
 * \brief    Callback for Receiver data handler of thread
 ******************************************************************************/
static void *rxdatahandler_callback(void *threadID) {
  int tid, ret = 0;;
  struct pollfd poll_fd[1];
  tid = (int)(uintptr_t)threadID;

  if (DEBUG_OUT > 1) printf("Thread Callback:%d\n",tid);

  for (;;) {
     //TODO: rx messages from CODU server and wait for terminate command
      poll_fd[0].fd = lps[tid].devicesockfd;
      poll_fd[0].events = POLLIN | POLLRDNORM;
      poll_fd[0].revents = 0;
            
      for(;;)
      {
            ret = poll(poll_fd, 1, -1);
            if (ret > 0)
            {
                if (poll_fd[0].revents & POLLIN || poll_fd[0].revents & POLLRDNORM)
                {
                    poll_fd[0].revents = 0;
                    // Read data
                    readdatafromSock(lps[tid].devicesockfd, tid);
                }
            }    
        }
    }
    return 0;
}


//***************************************************************************
//
// Function call to Get the parameter from the device using message
//  
//*****************************************************************************
bool GetParameter(DEVID deviceID, int GetParam)
{
    BYTE VNX_param[6] = {0, 0, 0, 0, 0, 0};

    if(deviceID >=0)
    {
        // Check Device open or not
        if (!isDeviceSockOpen(deviceID))
            return FALSE;

        if (TestMode) return TRUE; // Test mode, no need to send anything

        if (!SendReport(deviceID, GetParam, VNX_param, 0)) {
          return FALSE;
        }       
    }
    else return FALSE;
    return  TRUE;
}



//***************************************************************************
//
// This function is called to add an LPS device to the set of possible devices
// The IP address must be in the IPv4 format: e.g. 192.168.100.1
//
//***************************************************************************
int fnLPS_AddLPSDevice(char* deviceIP)
{
    int i;

    // test if the new IP address is in a valid format
    if (!is_valid_ip(deviceIP))
        return BAD_PARAMETER;

    // test if this IP address is already in our table of devices
    // if so, we just return success - this allows a calling application to
    // add devices without worrying about if a device was already added
    // we don't put devices in lps[0]

    for (i = 1; i < MAXDEVICES; i++)
    {
        if (strcmp(lps[i].deviceip, deviceIP) == 0)
            return STATUS_OK;
    }

    // look for an empty slot to store the new IP address in
    for (i = 1; i < MAXDEVICES; i++)
    {
        if (lps[i].deviceip[0] == '\0')
        {
            strcpy(lps[i].deviceip, deviceIP);
            lps[i].ModelName[0] = '\0';     // put a null string in each model name field
            lps[i].DevType = 0;
            lps[i].SerialNumber = 0;        // clear the serial number
            lps[i].devicesockfd = 0;        // clear the device handles
            lps[i].deviceready = 0;
            lps[i].responseready = 0;       // Device Response ready status

            for (int j = 0; j < CHANNEL_MAX; ++j) {
                lps[i].DevStatus[j] = 0;
            }

            lpsdevcount++;
            return STATUS_OK;
        }
    }

    // we did not find an empty slot in our table, not much we can do
    // but report the error
    return DEV_TABLE_FULL;

}

//***************************************************************************
//
// This function is called to remove an LPS device from the set of possible devices
// and make its slot available. The device must not be open
//
//***************************************************************************
int fnLPS_RemoveLPSDevice(char* deviceIP)
{
    int i;

    // test if the new IP address is in a valid format
    if (!is_valid_ip(deviceIP))
        return BAD_PARAMETER;

    // look for this IP address in our table of devices
    // if we find it, and the device is not opened, we will remove it
    for (i = 1; i < MAXDEVICES; i++)
    {
        if (strcmp(lps[i].deviceip, deviceIP) == 0)
        {
            if (lps[i].DevStatus[0] & DEV_OPENED)
                return DEVICE_NOT_READY;
            else
            {
                // we can go ahead and remove the device
                lps[i].deviceip[0] = '\0';
                lps[i].ModelName[0] = '\0';
                lps[i].DevType = 0;
                lps[i].SerialNumber = 0;
                lps[i].devicesockfd = 0;        // clear the device handles
                lps[i].deviceready = 0;
                lps[i].responseready = 0;       // Device Response ready status

                for (int j = 0; j < CHANNEL_MAX; ++j) {
                    lps[i].DevStatus[j] = 0;
                }

                lpsdevcount--;
                return STATUS_OK;
            }
        }
    }

    // if we did not find the IP address, we got a bad parameter
    return BAD_PARAMETER;
}

//***************************************************************************
// 
// Function to find devices and return number of devices that are connected
//
//***************************************************************************
int fnLPS_GetNumDevices(void)
{
    int NumDevices = 0;
    int i;
    int status;

    // See how many devices we can find, or have found before
    if (TestMode) {
        // construct a fake device

        lps[1].SerialNumber = 55102;
        lps[1].DevStatus[0] = lps[1].DevStatus[0] | DEV_CONNECTED;
        lps[1].DevType = LPS_802_8;
        lps[1].MinPhase = 0;			// 0 degrees is always the minimum
        lps[1].MaxPhase = 7200;			// 360 degrees in .05 degree units
        lps[1].UnitScale = 1;			// for consistency...
        lps[1].NumChannels = 8;
        lps[1].ProfileMaxLength = 1000;
        lps[1].MinPhaseStep = 20;
        lps[1].MinFrequency = 40000;	// 4000 MHz
        lps[1].MaxFrequency = 80000;	// 8000 MHz
        lps[1].deviceready = true;
        // Set some default starting values
        for (int ch = 1; ch <= 8; ch++) {
          lps[1].PhaseAngle[ch] = 0;			// Start with phase angle 0 degrees
          lps[1].WorkingFrequency[ch] = 40000;	// Start with 4000 MHz
          lps[1].PhaseStep[ch] = 20;		// 1 degree in .05 degree units
          lps[1].PhaseStep2[ch] = 20;		// 1 degree in .05 degree units
          lps[1].RampStart[ch] = 0;		// Start at 0 degrees
          lps[1].RampStop[ch] = 7200;		// 360 degrees in .05 degree units
          lps[1].DwellTime[ch] = 1000;		// 1 second dwell time
          lps[1].DwellTime2[ch] = 1;		// 1 ms dwell time two
          lps[1].IdleTime[ch] = 0;			// 0 seconds idle time
          lps[1].HoldTime[ch] = 0;			// 0 seconds hold time
          lps[1].ProfileCount[ch] = 50;	// 50 element profile
          lps[1].ProfileDwellTime[ch] = 1;	// 1 ms profile dwell time
          lps[1].ProfileIdleTime[ch] = 0; 	// 0 seconds profile idle time
        }


        strcpy (lps[1].ModelName, "LPS-802-8");
    }
    else
    {
        // go look for some real hardware
        status = FindVNXDevices();
        if (status < 0) return BAD_HID_IO;          // If we fail due to some OS issue we at least try to tell the user about it
    }

    // Total up the number of devices we have
    for (i = 1; i < MAXDEVICES; i++) {

        if (lps[i].DevStatus[0] & DEV_CONNECTED) NumDevices++;
    }

    return NumDevices;
}

//***************************************************************************
// 
// Get the number of channels for the selected device
// 
//***************************************************************************
int fnLPS_GetNumChannels(DEVID deviceID) {
  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (deviceID > 0) {
    GetParameter(deviceID, VNX_MAXCHANNEL);
  }
  return lps[deviceID].NumChannels;
}

//***************************************************************************
// 
// ------------ Get info for the connected devices --------------------
//
// ActiveDevices must be an array large enough to hold NumDevices DEVIDs
// The function returns the number of DEVIDs in the array
// 
//***************************************************************************
int fnLPS_GetDevInfo(DEVID *ActiveDevices) {
  int i;
  int NumDevices = 0;

  if ( ActiveDevices == NULL) return 0; // bad array pointer, no place to put the DEVIDs

  for (i = 1; i < MAXDEVICES; i++){ // NB -- we never put an active device in lps[0] - so DEVIDs start at 1
    if (lps[i].DevStatus[0] & DEV_CONNECTED) {
      ActiveDevices[NumDevices] = i;
      NumDevices++;
    }
  }

  return NumDevices;
}

//***************************************************************************
//
// Function call to Initialize the LPS Device and Open the socket connection
// to that device for communication 
//  
//*****************************************************************************
int fnLPS_InitDevice(DEVID deviceID) {
    int sockfd;
    int devid;
    int err;
    unsigned int index;
    unsigned int i;

    // Check whether already the socket open for that device
//  if(isDeviceSockOpen(deviceip))
//      return STATUS_ERROR;

    if (TestMode){    
        lps[deviceID].devicesockfd = 1234;
    }
    else {
        // Check if the device is connected
        if (!(lps[deviceID].DevStatus[0] & DEV_CONNECTED)) {
            return DEVICE_NOT_READY;
        }

        // Clear the device cache before continuing
        // This ensures that we have no leftover data from previous operations
        ClearDevCache(deviceID, CHANNEL_MAX);

        if (!is_valid_ip(lps[deviceID].deviceip))
            return BAD_IP_ADDRESS;

        // Get the ip address
        char deviceip[MAX_NETBUFF];
        strcpy(deviceip, lps[deviceID].deviceip);

        // Go ahead and open a handle to the hardware
        sockfd = vnxOpenDevice(deviceip);
        if (sockfd == DEVICE_NOT_READY) {
            return DEVICE_NOT_READY;
        }
        lps[deviceID].devicesockfd = sockfd;

 //       printf("Socketfd:%d\n",sockfd);
        // Create a device threads for reading the data from the device using sockets
        // err = CreateThread(NULL, 0, rxdatahandler_callback, (void*)(uintptr_t)deviceID, 0, &threads[deviceID]);
        err = pthread_create(&threads[deviceID], NULL, rxdatahandler_callback, (void*)(uintptr_t)deviceID);

        // Set a default channel mask
        lps[deviceID].ChMask = 0x01;

        // Get the current status details of the device parameter
        for(index=0; index < sizeof(STATUS_QUERY_ID); index++)
        {
            //printf("INDEX %d\n", index);

            int cmd = STATUS_QUERY_ID[index];

            if (!GetParameter(deviceID, cmd))
                return BAD_HID_IO;
        }

        // Wait for device to be ready before returing from init call
        while(fnLPS_CheckDeviceReady(deviceID));
    } // end of real device open process case

    // update the status for the device
    lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] | DEV_OPENED;

    // if we got here everything worked OK
    return STATUS_OK;
}


//***************************************************************************
//
// Function call to close the socket of the open device
//  
//*****************************************************************************
int fnLPS_CloseDevice(DEVID deviceID) {
    int sockfd;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    // Check Device open
    if(isDeviceSockOpen(deviceID))
    {
        // If Device exists then read the data 
        if(deviceID >= 0)
        {
            if (!TestMode){
                sockfd = lps[deviceID].devicesockfd;
                close(sockfd);
            }

            // And clean up for next time
            ClearDevCache(deviceID, CHANNEL_MAX);  // clear the cached variables to the unknown state
            lps[deviceID].DevType = 0;
            lps[deviceID].devicesockfd = 0;        // clear the device handles
            lps[deviceID].deviceready = 0;
            lps[deviceID].responseready = 0;       // Device Response ready status

            for (int j = 0; j < CHANNEL_MAX; ++j) {
                lps[deviceID].DevStatus[j] = 0;
            }

            lps[deviceID].DevStatus[0] |= DEV_CONNECTED; // Maintain connected status
        }
        else
            return INVALID_DEVID;
    }
    else
        return DEVICE_NOT_READY;
    return STATUS_OK;

}


//***************************************************************************
//
// Function call to Check Device is Ready or not
//  
//*****************************************************************************
int fnLPS_CheckDeviceReady(DEVID deviceID) {

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if((deviceID >= 0) && (lps[deviceID].deviceready !=0))
    {
        return STATUS_OK;
    }
    else
        return DEVICE_NOT_READY;
}

// Get the device's status - this function can be used for an open or closed device.
int fnLPS_GetDeviceStatus(DEVID deviceID) {
	int ChannelStatusMask = (PROFILE_ACTIVE | SWP_BIDIRECTIONAL | SWP_REPEAT | SWP_UP | SWP_ACTIVE);
	int ch = 0;

  if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

  // The V2 devices have enhanced status reports
  // Other devices we will still return status but it may not be very informative...
  // it will still tell you if the device is connected and open
  if (CheckV2Features(deviceID)) {
    if (!GetDeviceStatus(deviceID)) {
        return BAD_HID_IO;
    }

    if (status_empty) {
        return DATA_UNAVAILABLE;
    }
  }

  if (lps[deviceID].NumChannels > 1){
    ch = lps[deviceID].Channel;
    return (lps[deviceID].DevStatus[0] & ~ChannelStatusMask) | (lps[deviceID].DevStatus[ch] & ChannelStatusMask);
  }
  else
    return lps[deviceID].DevStatus[0];
}

//***************************************************************************
//
// Function call to Get Model name from the device (ASCII clients)
//  
//*****************************************************************************
int fnLPS_GetModelName(DEVID deviceID, char *ModelName) {
  int NumChars = 0;

  if (deviceID >= MAXDEVICES){
    return INVALID_DEVID;
  }

  NumChars = strlen(lps[deviceID].ModelName);
  // If NULL result pointer, just return the number of chars in the name
  if ( ModelName == NULL) return NumChars;
  strcpy(ModelName, lps[deviceID].ModelName);

  return NumChars;
}

//***************************************************************************
//
// Function call to Get Serial# from the device
//  
//*****************************************************************************
int fnLPS_GetSerialNumber(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].SerialNumber;
}

//***************************************************************************
//
// Function call to Get IP Mode from the device
//  
//*****************************************************************************
LVSTATUS fnLPS_GetIPMode(DEVID deviceID, int* respdata){

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPMODE);
        *respdata = lps[deviceID].ipmode;
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get IP Address from the device
//  
//*****************************************************************************
LVSTATUS fnLPS_GetIPAddress(DEVID deviceID, char* respdata) {
    
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPADDR);
        strcpy(respdata, lps[deviceID].ipaddress);
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Netmask from the device
//  
//*****************************************************************************
LVSTATUS fnLPS_GetNetmask(DEVID deviceID, char* respdata){
    
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPMASK);
        strcpy(respdata, lps[deviceID].netmask);
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Gateway from the device
//  
//*****************************************************************************
LVSTATUS fnLPS_GetGateway(DEVID deviceID, char* respdata){
    
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPGW);
        strcpy(respdata, lps[deviceID].gateway);
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}


// Functions to set parameters

// Set the phase angle - the function uses 1 degree units
LVSTATUS fnLPS_SetPhaseAngle(DEVID deviceID, int phase) {

	int phase_20 = phase * 20;	// convert to .05 degree units
	int tmp_phase = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	LockDev(deviceID, TRUE);

	int old_phase = lps[deviceID].PhaseAngle[lps[deviceID].Channel];

	if ((phase_20 >= lps[deviceID].MinPhase) && (phase_20 <= lps[deviceID].MaxPhase)) {
	  lps[deviceID].PhaseAngle[lps[deviceID].Channel] = phase_20;
	  if (TestMode){
	    LockDev(deviceID, FALSE);
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	  }
	} else {
	  LockDev(deviceID, FALSE);
	  return BAD_PARAMETER;
	}

	// the phase angle value is OK, lets send it to the hardware

	if (lps[deviceID].UnitScale > 0)
		tmp_phase = lps[deviceID].PhaseAngle[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_phase;

	if (!SendReport(deviceID, VNX_PHASEANGLE | VNX_SET, ptr, 4)) {
	  lps[deviceID].PhaseAngle[lps[deviceID].Channel] = old_phase;
	  LockDev(deviceID, FALSE);
	  return BAD_HID_IO;
	}

	LockDev(deviceID, FALSE);

	return STATUS_OK;
}


// Set the phase angle for a quad (or other multi-channel) phase shifter - the function uses 1 degree units
LVSTATUS fnLPS_SetPhaseAngleQ(DEVID deviceID, int phase, int channel) {

	LVSTATUS SCResult;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	// set the channel first
	SCResult = (LVSTATUS) SetChannel(deviceID, channel);
		if ( SCResult != STATUS_OK) return SCResult;

	return fnLPS_SetPhaseAngle(deviceID, phase);
}

LVSTATUS fnLPS_SetPhaseAngleMC(DEVID deviceID, int phase, unsigned long long channelmask) {
	unsigned int i;
	int ScaledPhaseAngle = 0;
	int cmdlen;
	unsigned long long tmp64;
	unsigned int chmask;				// the channel mask for the current logical bank
	unsigned int logicalbankid;			// the current logical bank id, starting at 1 and increasing for each bank we have
	unsigned int num_banks;				// used to loop over the number of active banks we have in out mask
	unsigned int ch;					// our internal channel index used for updating caches
	unsigned int tmask;					// a test mask
	int old_phaseangle[64];				// somewhere to put the old phase angle values in case we hvae to back out due to an IO error
	BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

	phase = phase * 20;	// convert phase angle to .05 degree units

	if (deviceID >= MAXDEVICES)
	{
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID))
	{
		return DEVICE_NOT_READY;
	}

	// check to see if the phase angle value is reasonable
	if ((phase < lps[deviceID].MinPhase) || (phase > lps[deviceID].MaxPhase))
	{
		return BAD_PARAMETER;
	}

	// generate our scaled value for this hardware
	if (lps[deviceID].UnitScale > 0)
		ScaledPhaseAngle = phase / lps[deviceID].UnitScale;

	// build the parts of the command packet that don't change across banks
	VNX_command[0] = (BYTE)(ScaledPhaseAngle & 0x000000ff);			// low byte of phase angle in .05 degree device units
	VNX_command[1] = (BYTE)((ScaledPhaseAngle & 0x0000ff00) >> 8);	// high byte of phase angle

	// our commands with logical bank and channel addresses have a 6 byte payload
	// but in case somebody runs this function with an old single channel device, I'll generate that command length format too
	if (CheckHiRes(deviceID))
	{
		cmdlen = 6;				// command length is set to 6 now, although some instances of this command being sent with length = 4 probably exist
		// the firmware doesn't care
	}
	else
	{
		cmdlen = 1;				// this was 4 in earlier versions, should be 1, firmware doesn't care.
	}

	// test our channel mask and clip it to our device's maximum number of channels
	// we don't return an error since applications level code might just use "ALL" as 
	// a quick way of setting every channel
	tmp64 = 0xffffffffffffffff >> (64 - lps[deviceID].NumChannels);
	channelmask &= tmp64;

	// get rid of an easy error case, nothing to do if we don't have any channel bits set
	// but let the user know their channel mask parameter was invalid for the device
	if (channelmask == 0)
	{
		return BAD_PARAMETER;
	}

	// determine how many logical banks our device has
	num_banks = lps[deviceID].NumChannels / 8;
	if ((lps[deviceID].NumChannels % 8) != 0)
	{
		num_banks++;
	}

	// loop over all of our logical banks, sending down the multiple channel set command for that bank and
	// saving the corresponding attenuation values for each channel

	LockDev(deviceID, TRUE);

	for (i = 0; i < num_banks; i++)
	{
		// select the 8 bit wide mask for the bank
		chmask = (channelmask >> (i * 8)) & 0xFF;

		// if we find an empty bank just skip over it
		if (!chmask) continue;

		// save a copy of the existing attenuation values for this logical bank's active channels and
		// update the cached attenuation value
		tmask = 1;
		for (ch = i * 8; ch < (i * 8 + 8); ch++)
		{
			if (chmask & tmask)
			{
				old_phaseangle[ch] = lps[deviceID].PhaseAngle[ch];
				lps[deviceID].PhaseAngle[ch] = phase;
			}
			tmask <<= 1;
		}

		// if we are in test mode we don't need to update the actual hardware
		if (TestMode) continue;

		// time to send the command for this logical bank to the device
		if (Trace_Out > 2) printf("SetAttenMC: DeviceID = %d, Internal BankID = %d, chmask = %d. Attenuation = %d\n", deviceID, i, chmask, ScaledPhaseAngle);

		VNX_command[4] = (BYTE)(i + 1);		// the logical bank ID is sent as a 1 to N value
		VNX_command[5] = (BYTE)chmask;		// the logical bank's active channel mask

		if (!SendReport(deviceID, VNX_PHASEANGLE | VNX_SET, VNX_command, 6))
		{
			// something failed in the I/O process so we will try to back out as best we can and bail out
			tmask = 1;
			for (ch = i * 8; ch < (i * 8 + 8); ch++)
			{
				if (chmask & tmask)
				{
					lps[deviceID].PhaseAngle[ch] = old_phaseangle[ch];
				}
				tmask <<= 1;
			}

			LockDev(deviceID, FALSE);
			return BAD_HID_IO;
	  }

		// go back and do the next logical bank
  }

  // it looks like we successfully sent all of the bank specific commands so return success
  LockDev(deviceID, FALSE);
  return STATUS_OK;
}

// Set the phase angle ramp start value
LVSTATUS fnLPS_SetRampStart(DEVID deviceID, int rampstart) {
	int rampstart_20 = rampstart * 20;	// convert to .05 degree units
	int tmp_rampstart = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_rampstart = lps[deviceID].RampStart[lps[deviceID].Channel];

	if ((rampstart_20 >= lps[deviceID].MinPhase) && (rampstart_20 <= lps[deviceID].MaxPhase)) {
	  lps[deviceID].RampStart[lps[deviceID].Channel] = rampstart_20;
	  if (TestMode){
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	  }
	} else {
	  return BAD_PARAMETER;
	}

	// the ramp start value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
		tmp_rampstart = lps[deviceID].RampStart[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_rampstart;

	if (!SendReport(deviceID, VNX_ASTART | VNX_SET, ptr, 4)){
	  lps[deviceID].RampStart[lps[deviceID].Channel] = old_rampstart;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the phase angle ramp end value
LVSTATUS fnLPS_SetRampEnd(DEVID deviceID, int rampstop) {
	int rampstop_20 = rampstop * 20;	// convert to .05 degree units
	int tmp_rampstop = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_rampstop = lps[deviceID].RampStop[lps[deviceID].Channel];

	if ((rampstop_20 >= lps[deviceID].MinPhase) && (rampstop_20 <= lps[deviceID].MaxPhase)){

	  lps[deviceID].RampStop[lps[deviceID].Channel] = rampstop_20;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}
	// the ramp end value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
	  tmp_rampstop = lps[deviceID].RampStop[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_rampstop;

	if (!SendReport(deviceID, VNX_ASTOP | VNX_SET, ptr, 4)) {
	  lps[deviceID].RampStop[lps[deviceID].Channel] = old_rampstop;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the phase angle ramp step size for the first phase
LVSTATUS fnLPS_SetPhaseAngleStep(DEVID deviceID, int phasestep) {
	int phasestep_20 = phasestep * 20;	// convert to .05 degree units
	int tmp_phasestep = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_phasestep = lps[deviceID].PhaseStep[lps[deviceID].Channel];

	if (phasestep_20 < (lps[deviceID].MaxPhase - lps[deviceID].MinPhase)) {
	  lps[deviceID].PhaseStep[lps[deviceID].Channel] = phasestep_20;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}
	// the phase ramp step value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
		tmp_phasestep = lps[deviceID].PhaseStep[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_phasestep;

	if (!SendReport(deviceID, VNX_ASTEP | VNX_SET, ptr, 4)) {
		lps[deviceID].PhaseStep[lps[deviceID].Channel] = old_phasestep;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the phase angle ramp step size for the second phase of the ramp
//
LVSTATUS fnLPS_SetPhaseAngleStepTwo(DEVID deviceID, int phasestep2) {
	int phasestep2_20 = phasestep2 * 20;	// convert to .05 degree units
	int tmp_phasestep2 = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_phasestep2 = lps[deviceID].PhaseStep2[lps[deviceID].Channel];

	if (phasestep2_20 < (lps[deviceID].MaxPhase - lps[deviceID].MinPhase)) {
	  lps[deviceID].PhaseStep2[lps[deviceID].Channel] = phasestep2_20;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}
	// the phase ramp step value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
		tmp_phasestep2 = lps[deviceID].PhaseStep2[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_phasestep2;

	if (!SendReport(deviceID, VNX_ASTEP2 | VNX_SET, ptr, 4)) {
		lps[deviceID].PhaseStep2[lps[deviceID].Channel] = old_phasestep2;
		return BAD_HID_IO;
	}
	return STATUS_OK;
}

// Functions to set parameters
// Channel is 1 to 4 or 1 to 8, or 1 to N for expandable devices at the API level

int SetChannel(DEVID deviceID, int channel)
{
	BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (deviceID >= MAXDEVICES){
    return INVALID_DEVID;
  }

  if (!isDeviceSockOpen(deviceID)){
    return DEVICE_NOT_READY;
  }

  if (channel < 1 || channel > lps[deviceID].NumChannels){
    return BAD_PARAMETER;
  }
	// we use zero based channel numbers internally, the API is 1 based, as is the new format command
	lps[deviceID].Channel = channel - 1;

  if (lps[deviceID].Expandable && (channel > (lps[deviceID].BaseChannels))) {
		// the channel is located on an expansion module so we use the new format command
		VNX_command[0] = (char)channel;

		lps[deviceID].ChMask = 0x01 << (channel % 8);	// This value is most likely unused for a matrix device, but it represents the channel mask for the active bank
														// RD!!!! looks like it should be channel - 1 %8

  }
  else {
	// convert to our mask form and save the mask
	lps[deviceID].ChMask = 0x01 << lps[deviceID].Channel;
    VNX_command[5] = (char)lps[deviceID].ChMask;

  }
  if (TestMode) {
	return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
  }
    // -- send the channel selection to the phase shifter --
    if (!SendReport(deviceID, VNX_CHANNEL | VNX_SET, VNX_command, 6))
      return BAD_HID_IO;

  return STATUS_OK;
}

LVSTATUS fnLPS_SetChannel(DEVID deviceID, int channel)
{
	LVSTATUS result;
	result = SetChannel(deviceID, channel);
	if(!result)
	{
		usleep(700000);  // 700msec
	}
	
	return result;
}


// Set the working frequency for HiRes LPS devices, frequency is in 100KHz units
LVSTATUS fnLPS_SetWorkingFrequency(DEVID deviceID, int frequency) {
	int tmp_frequency = 0;
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_frequency = lps[deviceID].WorkingFrequency[ch];

	if (DEBUG_OUT > 0) {
	  printf("Min frequency is %d\r\n", lps[deviceID].MinFrequency);
	  printf("Max frequency is %d\r\n", lps[deviceID].MaxFrequency);
	  printf("User frequency to set is %d\r\n", frequency);
	}

	if ((frequency >= lps[deviceID].MinFrequency) && (frequency <= lps[deviceID].MaxFrequency)) {
	  lps[deviceID].WorkingFrequency[ch] = frequency;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the working frequency value is OK, lets send it to the hardware
	tmp_frequency = frequency;
	unsigned char *ptr = (unsigned char *) &tmp_frequency;

	if (DEBUG_OUT > 0) printf("setting frequency to %d\r\n", tmp_frequency);

	if (!SendReport(deviceID, VNX_FREQUENCY | VNX_SET, ptr, 4)) {
	  lps[deviceID].WorkingFrequency[ch] = old_frequency;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to dwell at each step on the first phase of the ramp
LVSTATUS fnLPS_SetDwellTime(DEVID deviceID, int dwelltime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_dwelltime = lps[deviceID].DwellTime[ch];

	if (dwelltime >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].DwellTime[ch] = dwelltime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].DwellTime[ch];

	if (!SendReport(deviceID, VNX_ADWELL | VNX_SET, ptr, 4)) {
	  lps[deviceID].DwellTime[ch] = old_dwelltime;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to dwell at each step on the second phase of the ramp
LVSTATUS fnLPS_SetDwellTimeTwo(DEVID deviceID, int dwelltime2) {
	int ch = 0;
	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_dwelltime2 = lps[deviceID].DwellTime2[ch];

	if (dwelltime2 >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].DwellTime2[ch] = dwelltime2;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].DwellTime2[ch];

	if (!SendReport(deviceID, VNX_ADWELL2 | VNX_SET, ptr, 4)) {
	  lps[deviceID].DwellTime2[ch] = old_dwelltime2;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}


// Set the time to wait at the end of the ramp
LVSTATUS fnLPS_SetIdleTime(DEVID deviceID, int idletime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_idletime = lps[deviceID].IdleTime[ch];

	if (idletime >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].IdleTime[ch] = idletime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the idle time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].IdleTime[ch];

	if (!SendReport(deviceID, VNX_AIDLE | VNX_SET, ptr, 4)) {
	  lps[deviceID].IdleTime[ch] = old_idletime;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to wait between the first and second phase of a bidirectional ramp
LVSTATUS fnLPS_SetHoldTime(DEVID deviceID, int holdtime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_holdtime = lps[deviceID].HoldTime[ch];

	if (holdtime >= 0) {		// holdtime can be zero
	  lps[deviceID].HoldTime[ch] = holdtime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the hold time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].HoldTime[ch];

	if (!SendReport(deviceID, VNX_AHOLD | VNX_SET, ptr, 4)) {
	  lps[deviceID].HoldTime[ch] = old_holdtime;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the ramp direction -- "up" is TRUE to ramp upwards
LVSTATUS fnLPS_SetRampDirection(DEVID deviceID, bool up) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	if (up)
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_DIRECTION;	// ramp or sweep direction up (bit == 0)
	else
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_DIRECTION;	// ramp or sweep direction downwards


	return STATUS_OK;
}

// Set the ramp mode -- mode = TRUE for repeated ramp, FALSE for one time ramp
LVSTATUS fnLPS_SetRampMode(DEVID deviceID, bool mode) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	if (mode) {
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_CONTINUOUS;		// Repeated ramp or sweep
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_ONCE;
	} else {
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_ONCE;			// one time ramp or sweep
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_CONTINUOUS;
	}

	return STATUS_OK;
}

// Select bidirectional or unidirectional ramp mode
// TRUE for bidirectional ramp, FALSE for unidirectional ramp
LVSTATUS fnLPS_SetRampBidirectional(DEVID deviceID, bool bidir_enable) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

	if (bidir_enable)
	{
		lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_BIDIR;	// bidirectional ramp
	}
	else
	{
		lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_BIDIR;	// unidirectional ramp
	}

	return STATUS_OK;
}



// Start the ramp
LVSTATUS fnLPS_StartRamp(DEVID deviceID, bool go) {
	int icount;
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	BYTE VNX_ramp[] = {0, 0, 0, 0, 0, 0};

if (go)
	{
    icount = 3;
    VNX_ramp[0] = (char) lps[deviceID].Modebits[ch] & MODE_SWEEP;
		
	}
	else
	{
    icount = 3;
		VNX_ramp[0] = 0;
	}

	if (TestMode)
	  return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW

	if (DEBUG_OUT > 0) printf(" sending a ramp command = %x\n", VNX_ramp[0] );

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_ramp, icount))
	  return BAD_HID_IO;

	return STATUS_OK;
}

// --- Start a ramp on multiple channels ---
//
// chmask has a 1 bit set for every channel to be started
// mode is composed of the sweep command byte flags SWP_DIRECTION, SWP_CONTINUOUS, SWP_ONCE, and SWP_BIDIR
// the deferred flag is not supported, but reserved for future use. Set it to false when calling this function for compatibility with future library versions
//
LVSTATUS fnLPS_StartRampMC(DEVID deviceID, int mode, int chmask, bool deferred)
{
	int icount;
	unsigned int ch;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	BYTE VNX_ramp[] = { 0, 0, 0, 0, 0, 0 };

	if (mode != 0){
		VNX_ramp[0] = (BYTE)(mode & MODE_SWEEP);	// mode sweep for V2 includes the 0x10 bit
	}
	else {
		VNX_ramp[0] = 0;
	}

	if (TestMode) {
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}

	if (Trace_Out > 2)  printf(" sending a  multi channel ramp command = %x\n", VNX_ramp[0]);

	VNX_ramp[5] = (BYTE)chmask;

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_ramp, 6)){
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// ---------- profile control functions, only supported in V2 devices -----------

// set profile element, 1 degree units
// profile values are cached, and only read when user code asks for them, writes update the cache

LVSTATUS fnLPS_SetProfileElement(DEVID deviceID, int index, int phase) {
	int phase_20 = phase * 20;	// convert to .05 degree units
	int ch = 0;
	unsigned int bank;
	int old_element;
	int scaled_phase;
	int max_index;
	int cmdlen;
	int tmp;

	if (deviceID >= MAXDEVICES) {
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	max_index = lps[deviceID].ProfileMaxLength - 1;
	if (index < 0 || index > max_index)			// check if the index is valid
	{
		return BAD_PARAMETER;					// the profile index is invalid
	}

	ch = lps[deviceID].Channel;
	if (ch >= CHANNEL_MAX) ch = 0;				// just defending our arrays, should never happen

	bank = (unsigned int)(ch / 8) + 1;		// the logical bank is incremented before it is transmitted
											// this lets us determine if we have a logical bank in the command packet


	if ((phase_20 >= lps[deviceID].MinPhase) && (phase_20 <= lps[deviceID].MaxPhase)) {
		// keep the existing cached value in case our IO fails
		// and stash the new value since we are optimistic

		old_element = lps[deviceID].CachedProfileValue[ch];			// keep cached value in case IO fails

		tmp = index << 16;

		lps[deviceID].CachedProfileValue[ch] =  tmp | (phase_20 & 0xFFFF);	// and stash the new value (high word is index, low word is value)

		if (TestMode) return STATUS_OK;		// update our caches, but don't talk to the real HW (the TestMode version does not simulate the profile)

	}
	else {
		return BAD_PARAMETER;							// the phase value is out of range
	}

	// -- our parameters are good, so lets send them to the hardware

	// -- scale our phase angle value to the units used by this device --
	// As of 6-20-16 we use the device's UnitScale parameter for scaling,
	// and now we are of course scaling from .05 degree units
	if (lps[deviceID].UnitScale > 0)
		scaled_phase = phase_20 / lps[deviceID].UnitScale;

	// -- construct the command --
	BYTE VNX_command[] = {0, 0, 0, 0, 0, 0, 0};

  VNX_command[0] = (BYTE)(scaled_phase & 0x00FF);			// Phase low byte
  VNX_command[1] = (BYTE)((scaled_phase >> 8) & 0x00FF);	// Phase high byte
  VNX_command[2] = (BYTE)(index & 0x00FF);				// Index low byte
  VNX_command[3] = (BYTE)((index >> 8) & 0x03);			// Index high bits (only bits 0–1 used)
  VNX_command[4] = (BYTE)bank;							// Logical bank number
  VNX_command[5] = (BYTE)lps[deviceID].ChMask;			// Logical channel mask

	if (!SendReport(deviceID, VNX_SETPROFILE | VNX_SET, VNX_command, 6)) {
		// our send failed, we'll leave the cache as we found it
		lps[deviceID].CachedProfileValue[ch] = old_element;
		return BAD_HID_IO;
	}
	return STATUS_OK;
}

// Set the time to remain at each phase angle value in a profile

LVSTATUS fnLPS_SetProfileDwellTime(DEVID deviceID, int dwelltime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

	int old_dwelltime = lps[deviceID].ProfileDwellTime[ch];

	if (dwelltime >= VNX_MIN_DWELLTIME) {

		lps[deviceID].ProfileDwellTime[ch] = dwelltime;
		if (TestMode) {
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}

	// the profile dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileDwellTime[ch];

	if (!SendReport(deviceID, VNX_PROFILEDWELL | VNX_SET, ptr, 4)){

		lps[deviceID].ProfileDwellTime[ch] = old_dwelltime;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to wait at the end of a repeating profile

LVSTATUS fnLPS_SetProfileIdleTime(DEVID deviceID, int idletime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;
	int old_idletime = lps[deviceID].ProfileIdleTime[ch];

	if (idletime >= VNX_MIN_DWELLTIME){

		lps[deviceID].ProfileIdleTime[ch] = idletime;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else
	{
		return BAD_PARAMETER;
	}

	// the profile idle time value is OK, lets send it to the hardware

	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileIdleTime[ch];


	if (!SendReport(deviceID, VNX_PROFILEIDLE | VNX_SET, ptr, 4)){

		lps[deviceID].ProfileIdleTime[ch] = old_idletime;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the number of elements in the profile
LVSTATUS fnLPS_SetProfileCount(DEVID deviceID, int profilecount) {
	int ch = 0;
	int max_count;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;
	int old_count = lps[deviceID].ProfileCount[ch];

	max_count = lps[deviceID].ProfileMaxLength;
	if (profilecount <= max_count && profilecount > 0){

		lps[deviceID].ProfileCount[ch] = profilecount;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else
	{
		return BAD_PARAMETER;
	}

	// the profile count value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileCount[ch];

	if (!SendReport(deviceID, VNX_PROFILECOUNT | VNX_SET, ptr, 4)){

		lps[deviceID].ProfileCount[ch] = old_count;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}


// Start the profile immediately
LVSTATUS fnLPS_StartProfile(DEVID deviceID, int mode) {

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	// -- NB: VNX_command[1] and VNX_command[2] must be 0 for immediate sweep start --
	BYTE VNX_command[] = {0, 0, 0, 0};

	if (mode != 0)	// mode is 1 for a single profile, 2 for a repeating profile
	{
		VNX_command[0] = (BYTE) ((mode & 0x00000003) | STATUS_PROFILE_ACTIVE);	// start the profile
	}
	else
	{
		VNX_command[0] = 0;		// stop the profile
	}

	if (TestMode){
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}


	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_command, 3))
	{
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// -- This function uses the multi-channel command capability of V2 firmware devices (also in some earlier devices)
//	  to launch a number of profiles concurrently
//	  chmask has 1 bits for all active channels.
//	  mode is 1 for a single profile, 2 for repeating profiles, and 0 to stop the profiles
//	  the delayed parameter is not supported at this time, set it to false for future compatibility
//
LVSTATUS fnLPS_StartProfileMC(DEVID deviceID, int mode, int chmask, bool delayed)
{
	if (deviceID >= MAXDEVICES) {
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)) {
		return DEVICE_NOT_READY;
	}

	// -- NB: VNX_command[1] and VNX_command[2] must be 0 for immediate sweep start --
	BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0};

	if (mode != 0) {	// mode is 1 for a single profile, 2 for a repeating profile
		VNX_command[0] = (BYTE)((mode & 0x00000003) | STATUS_PROFILE_ACTIVE);	// start the profile
	}
	else {
		VNX_command[0] = 0;		// stop the profile
	}

	if (TestMode) {
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}

	VNX_command[5] = (BYTE) chmask;

	if (Trace_Out > 2)  printf(" sending a multi channel start profile command = %x\n", VNX_command[0]);

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_command, 6)) {
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Save the user settings to flash for autonomous operation
LVSTATUS fnLPS_SaveSettings(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  if (TestMode)
    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW

  BYTE VNX_savesettings[] = {0x42, 0x55, 0x31}; //three byte key to unlock the user protection.

  if (!SendReport(deviceID, VNX_SAVEPAR | VNX_SET, VNX_savesettings, 3))
    return BAD_HID_IO;

  return STATUS_OK;
}

// ------------- Functions to get parameters ---------------------

// Get the working frequency for HiRes Phase Shifters in 100 KHz units
int fnLPS_GetWorkingFrequency(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].WorkingFrequency[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetWorkingFrequency(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].WorkingFrequency[ch];
}

// Get the phase angle setting in 1 degree units
int fnLPS_GetPhaseAngle(DEVID deviceID) {
  int ch = 0;
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  // always get the phase angle from the device since it might have changed
  if (!GetPhaseAngle(deviceID)){
    return BAD_HID_IO;
  }

  return lps[deviceID].PhaseAngle[ch] / 20;
}

// Get the ramp start phase angle level
int fnLPS_GetRampStart(DEVID deviceID) {
  int ch = 0;
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].RampStart[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetRampStart(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].RampStart[ch] / 20;
}

// Get the phase angle at the end of the ramp
int fnLPS_GetRampEnd(DEVID deviceID) {
  int ch = 0;
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].RampStop[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetRampStop(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].RampStop[ch] / 20;
}

// Get the time to dwell at each step along the first phase of the ramp
int fnLPS_GetDwellTime(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

    if (lps[deviceID].DwellTime[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetDwellTime(deviceID)){
			return BAD_HID_IO;
		}
	}
    return lps[deviceID].DwellTime[ch];
}

// Get the time to dwell at each step along the second phase of the ramp
int fnLPS_GetDwellTimeTwo(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!isDeviceSockOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

    if (lps[deviceID].DwellTime2[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetDwellTime2(deviceID)){
			return BAD_HID_IO;
		}
	}
    return lps[deviceID].DwellTime2[ch];
}


// Get the idle time to wait at the end of the ramp
int fnLPS_GetIdleTime(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].IdleTime[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetIdleTime(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].IdleTime[ch];
}

// Get the hold time to wait at the end of the first phase of the ramp
int fnLPS_GetHoldTime(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].HoldTime[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetHoldTime(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].HoldTime[ch];
}

// Get the size of the phase angle step for the first phase of the ramp in 1 degree units
int fnLPS_GetPhaseAngleStep(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

    if (lps[deviceID].PhaseStep[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetPhaseAngleStep(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].PhaseStep[ch] / 20;
}

int fnLPS_GetPhaseAngleStepTwo(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!isDeviceSockOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

    if (lps[deviceID].PhaseStep2[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetPhaseAngleStep2(deviceID)){
			return BAD_HID_IO;
		}
	}

  return lps[deviceID].PhaseStep2[ch] / 20;
}

// Get functions related to profiles
int fnLPS_GetProfileDwellTime(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;
	if (lps[deviceID].ProfileDwellTime[ch] == -1)	// we don't have a value cached so go get it
		{
			if (!GetProfileDwellTime(deviceID)) {
				return BAD_HID_IO;
			}
		}

	return lps[deviceID].ProfileDwellTime[ch];		// fetch the value for the current channel
}

int fnLPS_GetProfileIdleTime(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

	if (lps[deviceID].ProfileIdleTime[ch] == -1)	// we don't have a value cached
	{
		if (!GetProfileIdleTime(deviceID)) {
				return BAD_HID_IO;
		}
	}

	return lps[deviceID].ProfileIdleTime[ch];	// fetch the value for the current channel
}

int fnLPS_GetProfileCount(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

		if (lps[deviceID].ProfileCount[ch] == -1) {	// we don't have a value cached, go get one
			if (!GetProfileCount(deviceID)) {
				return BAD_HID_IO;
			}
		}

	return lps[deviceID].ProfileCount[ch];	// fetch the value for the current channel

}


// get profile element, 1 degree units
int fnLPS_GetProfileElement(DEVID deviceID, int index) {
	int ch = 0;
	int max_index;
	int tmp;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!isDeviceSockOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	max_index = lps[deviceID].ProfileMaxLength - 1;
	ch = lps[deviceID].Channel;

	if (index < 0 || index > max_index)	{	// check if the index is valid
		return BAD_PARAMETER;
	}
	else {
		// we keep a single profile element cache
		tmp = lps[deviceID].CachedProfileValue[ch] >> 16;		// we keep the index of the cached value in the high 16 bits

		if (tmp == index) return (lps[deviceID].CachedProfileValue[ch] & 0xFFFF) / 20; // we've got a value cached
		else {
			if (!GetProfileElement(deviceID, index)) {
				return BAD_HID_IO;
			}
			return (lps[deviceID].CachedProfileValue[ch] & 0xFFFF) / 20;		// we read the value from the HW and cached it.
		}
	}
}

int fnLPS_GetProfileIndex(DEVID deviceID){
	int ch = 0;

  if (deviceID >= MAXDEVICES) {
      return INVALID_DEVID;
  }

  if (!isDeviceSockOpen(deviceID)) {
      return DEVICE_NOT_READY;
  }

  if (!CheckV2Features(deviceID)) {
      return FEATURE_NOT_SUPPORTED;
  }

  ch = lps[deviceID].Channel;

  // We want to fetch this value every time since it may have been updated
  if (!GetProfileIndex(deviceID)) {
      return BAD_HID_IO;
  }

  return lps[deviceID].ProfileIndex[ch];	// fetch the value for the current channel
}

// Get the maximum profile length supported by the device
int fnLPS_GetProfileMaxLength(DEVID deviceID)
{
	if (deviceID >= MAXDEVICES)
		return INVALID_DEVID;

	return lps[deviceID].ProfileMaxLength;
}

// Get the maximum phase shift in 1 degree units
int fnLPS_GetMaxPhaseShift(DEVID deviceID) {

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].MaxPhase / 20;
}

// Get the minimum phase shift in 1 degree units
int fnLPS_GetMinPhaseShift(DEVID deviceID) {

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].MinPhase / 20;
}

// Get the resolution of the phase shifter, in 1 degree units
int fnLPS_GetMinPhaseStep(DEVID deviceID) {

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].MinPhaseStep / 20;
}

// get the minimum working frequency of the HiRes units
int fnLPS_GetMinWorkingFrequency(DEVID deviceID)
{
	if (deviceID >= MAXDEVICES){ return INVALID_DEVID;}

	return lps[deviceID].MinFrequency;	// even quad devices only have one frequency range for all channels
}

// get the maximum working frequency of the HiRes units
int fnLPS_GetMaxWorkingFrequency(DEVID deviceID)
{
	if (deviceID >= MAXDEVICES){ return INVALID_DEVID;}

	return lps[deviceID].MaxFrequency;		// even quad devices only have one frequency range for all channels
}

// Get a set of bits describing the features available in this phase shifter device
int fnLPS_GetFeatures(DEVID deviceID) {
	int temp = DEFAULT_FEATURES;

	if (deviceID >= MAXDEVICES){ return INVALID_DEVID;}

	else {
		temp = (HAS_HIRES | HAS_BIDIR_RAMPS | HAS_PROFILES);// HiRes has bidirectional ramps, profiles and HiRes
	}

	if (lps[deviceID].NumChannels == 4)    temp |= HAS_4CHANNELS;
  if (lps[deviceID].NumChannels == 8)    temp |= HAS_8CHANNELS;
	if (lps[deviceID].ProfileMaxLength == 1000) temp |= HAS_LONG_PROFILE;

	return temp;
}
